package com.thecodecollective.netbeans.completion;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Vector;

import org.eclipse.core.internal.runtime.Log;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowActionDelegate;
import org.eclipse.ui.texteditor.AbstractTextEditor;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.texteditor.ITextEditor;

import sun.misc.Cache;


/** 
 * This is the Completion action that simulates
 * the Netbeans Ctrl+K behavior.<br>
 * It finds the word prefix before the caret and then
 * scans all open documents in the current window for possible 
 * completions.
 * The word splitting is done using {@link java.lang.Character#isJavaIdentifierPart(char)} 
 * method.<br>
 * Each suggestion appears only once, and also there is always the empty
 * suggestion. Suggestions from the current editor
 * are preferred over suggestions from other editors.<br>
 * 
 * The completion process is aborted when the document changes
 * (not as the result of completion), or the caret moves
 * to a different position. However, if the caret returns to
 * the position where the last completion process left it,
 * the completion operation will resume. This can be considered
 * a bug.<br>
 * 
 * Considering latest accepted completions:<br>
 * Accepting a completion means pressing the completion key several times
 * and then doing something else, that would reset the completion state.
 * 
 * @author Genady, genadyb@inter.net.il
 * @author Jamie, jamie@thecodecollective.com
 */
public class CompletionAction implements IWorkbenchWindowActionDelegate {
	
	/** The window where the action appears */
	private IWorkbenchWindow activeWindow = null;
	
	private static final String COMPLETION_PLUGIN = "com.thecodecollective.netbeans.completion";
	
	private static Plugin plugin = null;

	/** 
	 * The completion state to continue
	 * the iteration over suggestions
	 */
	private CompletionState lastCompletion = null;

	/** The list of latest accepted completions */
	private LinkedList completionHistory;
	
	private static void log(String message, Throwable exception) {
		if (plugin == null) {
			plugin = Platform.getPlugin(COMPLETION_PLUGIN);
		}
		if (plugin != null) {
			plugin.getLog().log(new Status(Status.INFO, COMPLETION_PLUGIN, Status.OK, message, exception));
		}
	}
	
	private static void log(String message) {
		log(message, null);
	}

	/**
	 * This class represents the state of the last completion process.
	 * Each time the user moves to a new position and calles this action
	 * an instance of this inner classs is created  and saved in
	 * {@link #lastCompletion}.
	 */
	private class CompletionState implements IDocumentListener {
		
		/** The editor in which the completion is performed */
		final ITextEditor parentEditor;
		
		/** The list of suggestions */
		final String[] suggestions;
		
		final String prefix;
		
		/** The caret position at which we insert the suggestions */
		final int startOffset;
		
		/** The index of next suggestion string */
		int nextSuggestion;
		
		/** The length of the last suggestion string */
		int length;
		
		boolean valid = true;
		
		CompletionState(ITextEditor parent, String prefix, String[] suggestions, int startOffset) {
			parentEditor = parent;
			this.suggestions = suggestions;
			this.startOffset = startOffset;
			this.prefix = prefix;
			length = 0;
			nextSuggestion = 0;
			
			// reserve the place for the accepted completion of this 
			// completion action
			completionHistory.addFirst(null);
		}

		/**
		 * Perform the next completion.
		 */
		public void next() {
			IDocumentProvider provider = parentEditor.getDocumentProvider();
			IEditorInput input = parentEditor.getEditorInput();
			IDocument doc = provider.getDocument(input);
			doc.removeDocumentListener(this);
			
			try {
				doc.replace(startOffset, length, suggestions[nextSuggestion]);
			} catch (BadLocationException e) {
				// we should never get here
				throw new IllegalStateException(e.toString());
			}
			updateCompletionHistory(prefix + suggestions[nextSuggestion]);
			length = suggestions[nextSuggestion].length();
			parentEditor.getSelectionProvider().
					setSelection(new TextSelection(startOffset+length, 0));
			nextSuggestion = (nextSuggestion + 1) % suggestions.length;
			doc.addDocumentListener(this);
		}

		public void updateCompletionHistory(String accepted) {
			completionHistory.set(0, accepted);
		}
		
		/**
		 * @return Returns the editor in which the completion
		 * is performed.
		 */
		public ITextEditor getParentEditor() {
			return parentEditor;
		}
		
		/**
		 * Check whether the document modification timestamp
		 * and the caret position allow us to continue with 
		 * the completion
		 */
		public boolean isValid() {
			
			if (!valid) {
				return false;
			}
			
			IDocumentProvider provider = parentEditor.getDocumentProvider();
			IEditorInput input = parentEditor.getEditorInput();
			ISelectionProvider selProvider = parentEditor.getSelectionProvider();
			ITextSelection sel = (ITextSelection)selProvider.getSelection();
			return sel.getOffset() == startOffset + length;
		}
		
		/**
		 * @see IDocumentListener#documentAboutToBeChanged(DocumentEvent)
		 */
		public void documentAboutToBeChanged(DocumentEvent event) {
		}

		/**
		 * @see IDocumentListener#documentChanged(DocumentEvent)
		 */
		public void documentChanged(DocumentEvent event) {
			valid = false;
			event.getDocument().removeDocumentListener(this);
		}
		
		public void removeListener() {
			IDocumentProvider provider = parentEditor.getDocumentProvider();
			IEditorInput input = parentEditor.getEditorInput();
			IDocument doc = provider.getDocument(input);
			doc.removeDocumentListener(this);
		}
	}

	/**
	 * Find all words in a document that start with the specified prefix
	 * The results are placed in a vector of suggestions.
	 * The results vector contains unique suggestions, so
	 * new suggestion is not added if there is already such suggestion.
	 * This makes the operation complexity higher than needed,
	 * but we can live with it.<br>
	 * 
	 * The word boundaries are defined by characters which are
	 * not {@link Character#isJavaIdentifierPart(char)}.
	 */
	private static void getSuggestions(String document, String prefix, Vector results, int startIndex) {
		
		log("start index is " + startIndex);
		int index = 0;
		while ((index = document.indexOf(prefix, index)) != -1) {
			
			if (index !=0 && Character.isJavaIdentifierPart(document.charAt(index - 1))) {
				index += prefix.length();
				continue;
			}
			
			int firstChar = index + prefix.length();
			int lastChar = firstChar;
			
			while (lastChar < document.length() &&
				Character.isJavaIdentifierPart(document.charAt(lastChar))) 
			{
				lastChar++;
			}
			
			if (lastChar != firstChar) {
				String addition = document.substring(firstChar, lastChar);
				if (index < startIndex) {
					results.remove(addition);
					results.add(0, addition);
					log(index + " added first " + results);
				} else {
					if (!results.contains(addition)) {
						results.add(addition);
						log(index + " added last " + results);
					}
				}
			}
			index = lastChar;
		}
	}

	/**
	 * Make sure that the element at position 0,
	 * which was left by the last completion operation
	 * does not appear anywhere else
	 */
	private static void scrubHistory(LinkedList history) {
		try {
			String firstElement = (String) history.getFirst();
			history.subList(1, history.size()).remove(firstElement);
			if (history.size() > 100) {
				Iterator iter = history.listIterator(100);
				while (iter.hasNext()) {
					iter.next();
					iter.remove();
				}
			}
		} catch (NoSuchElementException e) {
			return;
		} catch (IndexOutOfBoundsException e) {
			return;
		}
	}

	/**
	 * Put latest accepted completions before other
	 * possible completions
	 */
	private static Vector considerLatestCompletions(List latest, List current, String prefix) {
		
		int prefixLength = prefix.length();
		
		// create a hash set of all suggestions for the prefix
		// put here complete strings, since "latest"
		// contains complete strings.
		HashSet currentSuggestionsSet = new HashSet();
		for (Iterator iter = current.iterator(); iter.hasNext(); ) {
			String s = (String) iter.next();
			currentSuggestionsSet.add(prefix + s);
		}
		
		Vector prioritizedList = new Vector(current.size());
		
		// first put the latest suggestions in the order they appear
		// and remove them, so we won't add them again
		for (Iterator iter = latest.iterator(); iter.hasNext();) {
			String s = (String) iter.next();
			if (currentSuggestionsSet.contains(s)) {
				currentSuggestionsSet.remove(s);
				prioritizedList.add(s.substring(prefixLength));
			}
		}
		
		// now add the rest of the completions in the same order
		// they initially appeared
		for (Iterator iter = current.iterator(); iter.hasNext();) {
			String s = (String) iter.next();
			s = prefix + s;
			if (currentSuggestionsSet.contains(s)) {
				currentSuggestionsSet.remove(s);
				prioritizedList.add(s.substring(prefixLength));
			}
		}
		
		return prioritizedList;
	}

	/**
	 * Create the array of suggestions. It scan all open text editors
	 * and prefers suggestion from the currently open editor.
	 * It also addes the empty suggestion at the end.
	 */
	public String[] getSuggestions(String prefix) {
		
		// Change the order of open editors, to make the active editor
		// to appear first.
		IEditorReference editorsArray[] = activeWindow.getActivePage().getEditorReferences();
		Vector editorsVector = new Vector();
		for (int i = 0; i < editorsArray.length; i++) {
			IEditorPart realEditor = editorsArray[i].getEditor(false);
			if (realEditor != null) {
				editorsVector.add(realEditor);
			}
		}
		editorsVector.remove(activeWindow.getActivePage().getActiveEditor());
		editorsVector.add(0, activeWindow.getActivePage().getActiveEditor());
		
		// collect the suggestions from all open text editors
		Vector suggestions = new Vector();
		for (int i = 0; i < editorsVector.size(); i++) {
			
			if (editorsVector.get(i) instanceof AbstractTextEditor) {
				AbstractTextEditor textEditor = (AbstractTextEditor) editorsVector.get(i);
				IDocument doc = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
				int startIndex = 0;
				if (i == 0) {
					ITextSelection textSelection = ((ITextSelection) textEditor.getSelectionProvider().getSelection());
                    startIndex = textSelection.getOffset();
				}
				getSuggestions(doc.get(), prefix, suggestions, startIndex);
				log(textEditor.getTitle() + ": " + suggestions);
			}
		}
		
		// put completions the user lately accepted before other
		// suggestions. Also clean the completion history from
		// possible duplicate entries and make sure it does 
		// not exceed the maximum allowed size
		scrubHistory(completionHistory);
		// suggestions = considerLatestCompletions(completionHistory, suggestions, prefix);
		
		suggestions.add(""); // empty suggestion
		return (String[])suggestions.toArray(new String[0]);
	}

	/**
	 * Return the part of a word before the caret.
	 * If the caret is not at a middle/end of a word, 
	 * returns null.
	 */
	public String getCurrentPrefix(ITextEditor textEditor) {
		IDocument doc = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
		ITextSelection selection = (ITextSelection)textEditor.getSelectionProvider().getSelection();
		if (selection.getLength() > 0) {
			return null;
		}
		int pos = selection.getOffset();
        String docText = doc.get();
        int prevNonAlpha = pos;
        while (prevNonAlpha > 0 && 
        	    Character.isJavaIdentifierPart(docText.charAt(prevNonAlpha-1))) 
        {
        	prevNonAlpha--;
        }
        if (prevNonAlpha != pos) {
            return docText.substring(prevNonAlpha, pos);
        } else {
        	return null;
        }
	}

	/** 
	 * Run the action. First check whether we can continue
	 * the previous completion operation. If not,
	 * create new completion operation and invoke it.
	 */		
	public void run(IAction proxyAction) {
	
		IEditorPart editor = activeWindow.getActivePage().getActiveEditor();
		if (lastCompletion != null && lastCompletion.isValid()
			&& editor == lastCompletion.getParentEditor()) 
		{
			lastCompletion.next();
		} else {
			if (lastCompletion != null) {
				lastCompletion.removeListener();
			}
						
			ITextEditor textEditor;
			if (editor instanceof ITextEditor) {
				textEditor = (ITextEditor)editor;
			} else {
				activeWindow.getShell().getDisplay().beep();
				return;
			}
			String prefix = getCurrentPrefix(textEditor);
			if (prefix == null) {
				activeWindow.getShell().getDisplay().beep();
				return;
			}
			String[] suggestions = getSuggestions(prefix);
			
			// if it is single empty suggestion
			if (suggestions.length == 1) {
				activeWindow.getShell().getDisplay().beep();
				return;
			}
			lastCompletion = new CompletionState(textEditor, prefix, suggestions, 
					((ITextSelection)textEditor.getSelectionProvider().getSelection()).getOffset());
			lastCompletion.next();
		}
	}

	/**
	 * If we change the active editor, clear the completion
	 * state.
	 */
	public void selectionChanged(IAction proxyAction, ISelection selection) {
		if (selection instanceof TextSelection) {
			proxyAction.setEnabled(true);
			return;
		}
		proxyAction.setEnabled(!selection.isEmpty() && 
			(activeWindow.getActivePage().getActivePart() instanceof ITextEditor));

		if (lastCompletion != null) {
			lastCompletion.removeListener();
		}
		lastCompletion = null;
	}
	
	
	/**
	 * @see IWorkbenchWindowActionDelegate#dispose()
	 */
	public void dispose() {
	}

	/**
	 * @see IWorkbenchWindowActionDelegate#init(org.eclipse.ui.IWorkbenchWindow)
	 */
	public void init(IWorkbenchWindow window) {
		activeWindow = window;
		completionHistory = new LinkedList();
	}
}


